/* ---------------------------------------------------------------------
 * ATtiny13: CrazyCart Throttle Filter
 * Datasheet for ATtiny13: http://www.atmel.com/images/doc2535.pdf
 * 
 * Pin configuration -
 * PB1/OC0B: output (Pin 6)
 * PB3: Input (Pin 2)
 * PB0: Brakeswitch (with pullup) (Pin 5)
 * PB2: Speedlimit switch (with pullup) (Pin 7)
 * -------------------------------------------------------------------*/

//sudo avrdude -c usbasp -p t13 -u -U flash:w:main.hex:a
//fuses: lfuse: 7A  hfuse: FB
 
// 9.6 MHz, built in resonator
#define F_CPU 9600000
#define OUT PB1 // PB1=OC0B

#define PIN_BRAKE PB0 //PB0 is brakeswitch
#define PIN_SPEED PB2 //PB2 is speedlimit switch

#define PIN_ENABLE PB4 //PB4

 
 
#include <avr/io.h>
#include <util/delay.h>
#include <stdint.h>
//#include <stdlib.h>


//hall low: 0.9V  high: 4.24V

#define MININPUT 47 //0% throttle adc input
#define MAXINPUT 216 //100% throttle adc input

#define MINTHROTTLE 127 //pwm output for 0% throttle
#define MAXTHROTTLE 255 //pwm output for 100% throttle
#define MAXTHROTTLE_LOW 150 //pwm maxthrottle if speed limited

#define BRAKECOMMAND 0 //pwm output for brake


void pin_setup(void)
{

    DDRB &= ~(1 << PIN_BRAKE); //input
    DDRB &= ~(1 << PIN_SPEED); //input  
    DDRB &= ~(1 << PIN_ENABLE); //input  

    PORTB |= (1<< PIN_BRAKE); //pullup
    PORTB |= (1<< PIN_SPEED); //pullup
    PORTB |= (1<< PIN_ENABLE); //pullup
}


void adc_setup (void)
{
    // Set the ADC input to PB3/ADC3 MUX=11
    ADMUX |= (1 << MUX0);
    ADMUX |= (1 << MUX1);
    ADMUX |= (1 << ADLAR);
 
    // Set the prescaler to clock/128 & enable ADC
    // At 9.6 MHz this is 75 kHz.
    // See ATtiny13 datasheet, Table 14.4.
    ADCSRA |= (1 << ADPS1) | (1 << ADPS0) | (1 << ADEN);
}
 
uint32_t adc_read (void)
{
    // Start the conversion
    ADCSRA |= (1 << ADSC);
 
    // Wait for it to finish
    while (ADCSRA & (1 << ADSC));
 
    return ADCH;
}
 
void pwm_setup (void)
{
    // OUT is an output.
    DDRB |= (1 << OUT);  

    // Set Timer 0 prescaler to clock/1.
    // At 9.6 MHz this is 1.2 MHz.
    // See ATtiny13 datasheet, Table 11.9.
    TCCR0B |= (1 << CS00);
 
    // Set to 'Fast PWM' mode
    TCCR0A |= (1 << WGM01) | (1 << WGM00);
 
    // Clear OC0B output on compare match, upwards counting.
    TCCR0A |= (1 << COM0B1);
}
 
void pwm_write (uint8_t val)
{
    OCR0B = val;
}

 
int main (void)
{

    uint32_t adc_in;
    uint32_t pwm_out=0;
    uint32_t adc_speed;
    uint8_t maxthrottle_variable=MAXTHROTTLE;

    pin_setup();
    adc_setup();
    pwm_setup();
    

    pwm_write( (uint8_t)pwm_out );
	   


    //MAINLOOP  
    while (1) {
        adc_in = adc_read();

	if (adc_in>MAXINPUT){
		adc_speed=maxthrottle_variable;
	}else if(adc_in<MININPUT){
		adc_speed=MINTHROTTLE;
	}else{
		adc_speed=(adc_in-MININPUT)* (maxthrottle_variable-MINTHROTTLE)/(MAXINPUT-MININPUT) + MINTHROTTLE;
	}

	if ( !(PINB & (1<<PIN_SPEED)) ) { //Speedswitch low
		maxthrottle_variable=MAXTHROTTLE_LOW;
	}else{
		maxthrottle_variable=MAXTHROTTLE;
	}
	
	if ( !(PINB & (1<<PIN_BRAKE)) ) { //Brake pin low
		pwm_out=BRAKECOMMAND; //brake pressed
	}else{
		pwm_out=adc_speed;
	}

	if ( !(PINB & (1<<PIN_ENABLE)) ) { //enable pin low, switch closed
		pwm_out=MINTHROTTLE;
	}


	pwm_write( (uint8_t)pwm_out );
	

        
    }
}
