/* ---------------------------------------------------------------------
 * ATtiny13: CrazyCart Throttle Filter
 * Datasheet for ATtiny13: http://www.atmel.com/images/doc2535.pdf
 * 
 * Pin configuration -
 * PB1/OC0B: output (Pin 6)
 * PB3: Input Accelerator (Pin 2)
 * PB4: Input Brake (Pin 3)
 * PB0: some unused switch (with pullup) (Pin 5)
 * PB2: Speedlimit switch (with pullup) (Pin 7)
 * -------------------------------------------------------------------*/

//sudo avrdude -c usbasp -p t13 -u -U flash:w:main.hex:a
//fuses: lfuse: 7A  hfuse: FB
 
// 9.6 MHz, built in resonator
#define F_CPU 9600000
#define OUT PB1 // PB1=OC0B

#define PIN_BRAKE PB2 //PB0 is brakeswitch (not used)
#define PIN_SPEED PB0 //PB2 is speedlimit switch

 
 
#include <avr/io.h>
#include <util/delay.h>
#include <stdint.h>
//#include <stdlib.h>


//hall low: 0.9V (47) high: 4.24V (216)

//gemessen, gas: 0.93V - 3.33V
//	bremse: 0.87V  (1V set) - 2.7V

#define MININPUT 47 //0% throttle adc input, accelerator
#define MAXINPUT 170 //100% throttle adc input, accelerator

#define MININPUT_BRAKE 51 //0% throttle adc input, brake
#define MAXINPUT_BRAKE 138 //100% throttle adc input, brake

#define MINTHROTTLE 110 //pwm output for 0% throttle  //was 84 for quickfix
#define MAXTHROTTLE 255 //pwm output for 100% throttle  //was 168 for quickfix
#define MINBRAKE 0 //pwm output for 0% brake
#define MAXBRAKE 110 //pwm output for full brake  //was 84 for quickfix
#define BRAKETHRES 105 //pwm threshold for brake to brake,  set between min and maxbrake. main threshold set by voltage swing. brakethres only for tolarances
//#define MAXTHROTTLE_LOW 150 //pwm maxthrottle if speed limited

//output pwm values are software set to maximum 3,3V (168), middle=84, because fail

//#define BRAKECOMMAND 0 //pwm output for brake


void pin_setup(void)
{

    DDRB &= ~(1 << PIN_BRAKE); //input
    DDRB &= ~(1 << PIN_SPEED); //input

    PORTB |= (1<< PIN_BRAKE); //pullup
    PORTB |= (1<< PIN_SPEED); //pullup
}


void adc_setup (void)
{
    // Set the ADC input to PB3/ADC3 MUX=11
    ADMUX |= (1 << MUX0);
    ADMUX |= (1 << MUX1); //preset mux

    ADMUX |= (1 << ADLAR); //8-bit precision from ADCH
    ADMUX |= (0 << REFS0); //Vcc as reference, default on attiny13
 
    // Set the prescaler to clock/128 & enable ADC
    // At 9.6 MHz this is 75 kHz.
    // See ATtiny13 datasheet, Table 14.4.
    ADCSRA |= (1 << ADPS1) | (1 << ADPS0) | (1 << ADEN);
}
 
uint32_t adc_read_accelerator (void)
{
    //ADMUX = (1 << MUX1) | (1 << MUX0); //Switch to PB3 
    ADMUX |= (1 << MUX0); //set mux0 to 1
    ADCSRA |= (1 << ADSC);     // Start the conversion
    while (ADCSRA & (1 << ADSC)); // Wait for it to finish
    return ADCH;
}

uint32_t adc_read_brake (void)
{
    //ADMUX = (1 << MUX1) | (0 << MUX0); //Switch to PB4
    ADMUX &= ~(1 << MUX0); //set mux0 to 0
    ADCSRA |= (1 << ADSC);     // Start the conversion
    while (ADCSRA & (1 << ADSC)); // Wait for it to finish
    return ADCH;
}
 
void pwm_setup (void)
{
    // OUT is an output.
    DDRB |= (1 << OUT);  

    // Set Timer 0 prescaler to clock/1.
    // At 9.6 MHz this is 1.2 MHz.
    // See ATtiny13 datasheet, Table 11.9.
    TCCR0B |= (1 << CS00);
 
    // Set to 'Fast PWM' mode
    TCCR0A |= (1 << WGM01) | (1 << WGM00);
 
    // Clear OC0B output on compare match, upwards counting.
    TCCR0A |= (1 << COM0B1);

    //pwm frequenz gemessen ca 40khz
}
 
void pwm_write (uint8_t val)
{
    OCR0B = val;
}

 
int main (void)
{

    uint32_t adc_in_accelerator;
    uint32_t adc_in_brake;
    uint32_t pwm_out=0;
    uint32_t adc_speed; //speed to set
    uint32_t adc_brake; //brake value to set, if brake pressed
    //uint8_t maxthrottle_variable=MAXTHROTTLE;

    pin_setup();
    adc_setup();
    pwm_setup();
    

    pwm_write( (uint8_t)pwm_out );
	   


    //MAINLOOP  
    while (1) {
        adc_in_accelerator = adc_read_accelerator();
	adc_in_brake = adc_read_brake();

	/*if ( !(PINB & (1<<PIN_SPEED)) ) { //Speedswitch low
		maxthrottle_variable=MAXTHROTTLE_LOW;
	}else{
		maxthrottle_variable=MAXTHROTTLE;
	}*/

	if (adc_in_accelerator>MAXINPUT){
		adc_speed=MAXTHROTTLE;
	}else if(adc_in_accelerator<MININPUT){
		adc_speed=MINTHROTTLE;
	}else{
		adc_speed=(adc_in_accelerator-MININPUT)* (MAXTHROTTLE-MINTHROTTLE)/(MAXINPUT-MININPUT) + MINTHROTTLE;
	}

	if (adc_in_brake>MAXINPUT_BRAKE){
		adc_brake=MAXBRAKE;
	}else if(adc_in_brake<MININPUT_BRAKE){
		adc_brake=MINBRAKE;
	}else{
		adc_brake=(adc_in_brake-MININPUT_BRAKE)* (MAXBRAKE-MINBRAKE)/(MAXINPUT_BRAKE-MININPUT_BRAKE) + MINBRAKE;
	}

	adc_brake=MAXBRAKE-(adc_brake-MINBRAKE) +MINBRAKE; //invert brake adc, because voltage gets higher when pressed
	
	/*if ( !(PINB & (1<<PIN_BRAKE)) ) { //Brake pin low
		if (pwm_out>MINTHROTTLE){ //if throttle was positive,
			pwm_out=MINTHROTTLE; //set to off
		}
		if (pwm_out>BRAKECOMMAND){
			pwm_out-=1;
			_delay_ms(10); //slowly decrease
		}
	}else{
		pwm_out=adc_speed;
	}*/

	if ( !(PINB & (1<<PIN_BRAKE)) ) { //Brake pin low
		pwm_write((uint8_t)MINBRAKE);
	}else{

		if (adc_brake<BRAKETHRES){ //set adc to brake value  
			//pwm_out=adc_brake;
			pwm_write( (uint8_t)adc_brake );
		}else{ //not braking, adc to throttle speed value
			//pwm_out=adc_speed;
			pwm_write( (uint8_t)adc_speed );
		}
	}


	//pwm_write( (uint8_t)pwm_out );
	

        
    }
}
