/* ---------------------------------------------------------------------
 * ATtiny13: CrazyCart Throttle Filter
 * Datasheet for ATtiny13: http://www.atmel.com/images/doc2535.pdf
 * 
 * Pin configuration -
 * PB1/OC0B: output (Pin 6)
 * PB3: Input (Pin 2)
 * PB0: DIP 0 (lsb)
 * PB2: DIP 1 (msb)
 * -------------------------------------------------------------------*/

//sudo avrdude -c usbasp -p t13 -u -U flash:w:main.hex:a
//fuses: lfuse: 7A  hfuse: FB
 
// 9.6 MHz, built in resonator
#define F_CPU 9600000
#define OUT PB1 // PB1=OC0B

#define DIP0 PB0 //DIP 0 is lsb
#define DIP1 PB2 //DIP 1 is msb

 
 
#include <avr/io.h>
#include <util/delay.h>
#include <stdint.h>
//#include <stdlib.h>
#define RAND_MAX 1024 //for noise, doesnt work??

//#define ALPHAMIN 2 //slow/if close together; low value=slow
#define ALPHAMAX 256  //fast

#define MINTHROTTLE 50

uint16_t rand=11; //random seed lkg



void dip_setup(void)
{

    DDRB &= ~(1 << DIP0); //input
    DDRB &= ~(1 << DIP1); //input  

    PORTB |= (1<< DIP0); //pullup
    PORTB |= (1<< DIP1); //pullup
}


void adc_setup (void)
{
    // Set the ADC input to PB3/ADC3 MUX=11
    ADMUX |= (1 << MUX0);
    ADMUX |= (1 << MUX1);
    ADMUX |= (1 << ADLAR);
 
    // Set the prescaler to clock/128 & enable ADC
    // At 9.6 MHz this is 75 kHz.
    // See ATtiny13 datasheet, Table 14.4.
    ADCSRA |= (1 << ADPS1) | (1 << ADPS0) | (1 << ADEN);
}
 
uint32_t adc_read (void)
{
    // Start the conversion
    ADCSRA |= (1 << ADSC);
 
    // Wait for it to finish
    while (ADCSRA & (1 << ADSC));
 
    return ADCH;
}
 
void pwm_setup (void)
{
    // OUT is an output.
    DDRB |= (1 << OUT);  

    // Set Timer 0 prescaler to clock/1.
    // At 9.6 MHz this is 1.2 MHz.
    // See ATtiny13 datasheet, Table 11.9.
    TCCR0B |= (1 << CS00);
 
    // Set to 'Fast PWM' mode
    TCCR0A |= (1 << WGM01) | (1 << WGM00);
 
    // Clear OC0B output on compare match, upwards counting.
    TCCR0A |= (1 << COM0B1);
}
 
void pwm_write (uint8_t val)
{
    OCR0B = val;
}

void lkg_next(void)
{
    rand=(rand*109+89)%256;
}
 
int main (void)
{
    uint16_t alpha_min=2;    
    uint16_t alpha_max=255;

    uint32_t adc_in;
    uint32_t pwm_out=0;
    uint32_t alpha=ALPHAMAX; //in 1/1024
    uint16_t noise;
    uint32_t pwm_dithered;

    
    dip_setup(); 
    adc_setup();
    pwm_setup();

    //Set Parameters by dip switches
	if ( !(PINB & (1<<DIP1)) ) { //DIP1 low
	        if ( !(PINB & (1<<DIP0)) ) { //DIP0 low
		   alpha_min=1;
		   alpha_max=64;
		}else{ //DIP0 high  //not grounded
		   alpha_min=2;
		   alpha_max=128;
		}
	}else{ //DIP1 high  //not grounded
		if ( !(PINB & (1<<DIP0)) ) { //DIP0 low
		   alpha_min=2;
		   alpha_max=256;
		}else{ //DIP0 high  //not grounded
		   alpha_min=64;
		   alpha_max=512;
		}
	}
	

    


    //MAINLOOP  
    while (1) {
        adc_in = adc_read();

	if (adc_in<=MINTHROTTLE){ //instant stop
		pwm_out=adc_in*1024;
	}

	if (adc_in>=pwm_dithered)
		alpha=adc_in-pwm_dithered;
	else 
		alpha=pwm_dithered-adc_in;

	
	

	alpha= alpha*(alpha_max-alpha_min)/256 + alpha_min; //linear auf alphamin alphamax mappen
	
	pwm_out=(adc_in*alpha) + (pwm_out*(1024-alpha)/1024);

	lkg_next();

	//noise =0;
	pwm_dithered = (pwm_out+(rand*4)); //add noise for dithering
	pwm_dithered /=1024;

	for (int i=0;i<10;i++){
		if (pwm_dithered>=255) //limit maximum because of noise
			pwm_dithered=255;
		
		pwm_write( (uint8_t)pwm_dithered );
		_delay_us(1000);
	}
	

        
    }
}
